﻿//=============================================================================
// Ssao.fx Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// Oblicza mapę SSAO.
//=============================================================================

cbuffer cbPerFrame
{
	float4x4 gViewToTexSpace; // Proj*Texture
	float4   gOffsetVectors[14];
	float4   gFrustumCorners[4];

	// Współrzędne dane w przestrzeni widoku.
	float    gOcclusionRadius    = 0.5f;
	float    gOcclusionFadeStart = 0.2f;
	float    gOcclusionFadeEnd   = 2.0f;
	float    gSurfaceEpsilon     = 0.05f;
};
 
// Danych nienumerycznych nie można zapisywać w obiekcie cbuffer.
Texture2D gNormalDepthMap;
Texture2D gRandomVecMap;
 
SamplerState samNormalDepth
{
	Filter = MIN_MAG_LINEAR_MIP_POINT;

	// Ustaw bardzo daleką wartość głębokości, jeśli pobierana jest próbka
	// spoza mapy normalnych/głębokości, aby zapobiec fałszywym okluzjom.
	AddressU = BORDER;
	AddressV = BORDER;
	BorderColor = float4(0.0f, 0.0f, 0.0f, 1e5f);
};

SamplerState samRandomVec
{
	Filter = MIN_MAG_LINEAR_MIP_POINT;
	AddressU  = WRAP;
    AddressV  = WRAP;
};

struct VertexIn
{
	float3 PosL            : POSITION;
	float3 ToFarPlaneIndex : NORMAL;
	float2 Tex             : TEXCOORD;
};

struct VertexOut
{
    float4 PosH       : SV_POSITION;
    float3 ToFarPlane : TEXCOORD0;
	float2 Tex        : TEXCOORD1;
};

VertexOut VS(VertexIn vin)
{
	VertexOut vout;
	
	// Już w przestrzeni NDC.
	vout.PosH = float4(vin.PosL, 1.0f);

	// Indeks tablicy rogów ostrosłupa widzenia zapisujemy we współrzędnej x normalnej.
	vout.ToFarPlane = gFrustumCorners[vin.ToFarPlaneIndex.x].xyz;

	// Przekaż do shadera pikseli.
	vout.Tex = vin.Tex;
	
    return vout;
}

// Określa stopień zasłonięcia punktu p przez punkt q jako funkcję distZ.
float OcclusionFunction(float distZ)
{
	//
	// Jeżeli wartość głębokości dla q znajduje się "za" wartością głębokości dla p, q nie może zasłaniać p. 
	// Ponadto jeżeli głębokość(q) i głębokość(p) są wystarczająco bliskie, 
	// zakładamy, że q nie może zasłaniać p, ponieważ q musi być przed p o wartość Epsilon.
	//
	// Do wyznaczenia okluzji używamy następującej funkcji:  
	// 
	//
	//       1.0     -------------\
	//               |           |  \
	//               |           |    \
	//               |           |      \ 
	//               |           |        \
	//               |           |          \
	//               |           |            \
	//  ------|------|-----------|-------------|---------|--> zv
	//        0     Eps          z0            z1        
	//
	
	float occlusion = 0.0f;
	if(distZ > gSurfaceEpsilon)
	{
		float fadeLength = gOcclusionFadeEnd - gOcclusionFadeStart;
		
		// Zmniejszaj liniowo okluzję z 1 do 0 wraz ze zmianą distZ
		// od gOcclusionFadeStart do gOcclusionFadeEnd.	
		occlusion = saturate( (gOcclusionFadeEnd-distZ)/fadeLength );
	}
	
	return occlusion;	
}

float4 PS(VertexOut pin, uniform int gSampleCount) : SV_Target
{
	// p -- punkt, dla którego obliczamy okluzję otoczenia.
	// n -- wektor normalny w p.
	// q -- losowe przesunięcie z p.
	// r -- potencjalny punkt zasłaniający p.

	// Pobierz wektor normalny i współrzędną z w przestrzeni widoku dla tego piksela. Współrzędne tekstur dla
	// narysowanego czworokąta wielkości ekranu są już w przestrzeni uv.
	float4 normalDepth = gNormalDepthMap.SampleLevel(samNormalDepth, pin.Tex, 0.0f);
 
	float3 n = normalDepth.xyz;
	float pz = normalDepth.w;

	//
	// Odtwórz położenie (x,y,z) w przestrzeni widoku.
	// Znajdź t, takie że p = t*pin.ToFarPlane.
	// p.z = t*pin.ToFarPlane.z
	// t = p.z / pin.ToFarPlane.z
	//
	float3 p = (pz/pin.ToFarPlane.z)*pin.ToFarPlane;
	
	// Pobierz losowy wektor i dokonaj odwzorowania [0,1] --> [-1, +1].
	float3 randVec = 2.0f*gRandomVecMap.SampleLevel(samRandomVec, 4.0f*pin.Tex, 0.0f).rgb - 1.0f;

	float occlusionSum = 0.0f;
	
	// Pobierz punkty sąsiadujące z p na hemisferze zorientowanej w kierunku n.
	[unroll]
	for(int i = 0; i < gSampleCount; ++i)
	{
		// Wektory przesunięcia są równomiernie rozmieszczone w ustalonych punktach
		// (tak aby nie skupiały się wszystkie, wskazując w jednym kierunku). Jeśli odbijemy je, korzystając z losowego wektora,
		// otrzymamy zbiór losowo, ale równomiernie rozmieszczonych wektorów przesunięcia.
		float3 offset = reflect(gOffsetVectors[i].xyz, randVec);
	
		// Odwróć wektor przesunięcia, jeżeli znajduje się za płaszczyzną określoną przez (p, n).
		float flip = sign( dot(offset, n) );
		
		// Pobierz punkt blisko p będący w promieniu okluzji.
		float3 q = p + flip * gOcclusionRadius * offset;
		
		// Rzutuj q i generuj współrzędne rzutowe tekstury.   
		float4 projQ = mul(float4(q, 1.0f), gViewToTexSpace);
		projQ /= projQ.w;

		// Znajdź najbliższą wartość głębokości wzdłuż promienia biegnącego z
		// oka do q (nie jest to głębokość q, ponieważ q to dowolny punkt w pobliżu p, który może się znajdować 
		// w pustej przestrzeni). Najbliższą głębokość odszukujemy w mapie głębokości.

		float rz = gNormalDepthMap.SampleLevel(samNormalDepth, projQ.xy, 0.0f).a;

		// Odtwórz położenie r = (rx,ry,rz) w przestrzeni widoku.
		// Wiemy, że r leży na promieniu wyznaczanym przez q, istnieje więc takie t, że r = t*q.
		// r.z = t*q.z ==> t = r.z / q.z

		float3 r = (rz / q.z) * q;
		
		//
		// Test zasłaniania p przez r.
		// * Iloczyn skalarny dot(n, normalize(r - p) określa, jak
		//     blisko przed płaszczyzną (p,n) znajduje się zasłaniający
		//     punkt r. Im bliżej się znajduje, tym większa waga
		//     jego okluzji. Zapobiega to również zacienieniu punktu przez samego siebie,
		//     co może się zdarzyć, gdyż punkt r na nachylonej płaszczyźnie (p,n) mógłby
		//     powodować fałszywą okluzję, ponieważ ma inną wartość głębokości względem oka niż p.
		//   * Wagę okluzji skalujemy na podstawie odległości punktu zasłaniającego od punktu, dla którego obliczamy
		//     okluzję. Jeśli r jest daleko od p, nie zasłania go.
		// 
		
		float distZ = p.z - r.z;
		float dp = max(dot(n, normalize(r - p)), 0.0f);
		float occlusion = dp * OcclusionFunction(distZ);
		
		occlusionSum += occlusion;
	}
	
	occlusionSum /= gSampleCount;
	
	float access = 1.0f - occlusionSum;

	// Zwiększ kontrast mapy SSAO.
	return saturate(pow(access, 4.0f));
}

technique11 Ssao
{
    pass P0
    {
		SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS(14) ) );
    }
}
 